﻿using System;
using System.Collections.Generic;
using System.Linq;

using Telerik.Web.UI;

using VeteransAffairs.Registries.BusinessManager;

public partial class PatientImageDetails : ImagingBasePage
{
    private readonly PatientImageDetailManager _patientImageDetailManager;

    private static readonly IEnumerable<string> _referenceImageBodyParts;
    private static readonly IEnumerable<string> _referenceImageFragmentCounts;
    private static readonly IEnumerable<string> _referenceImageReasons;
    private static readonly IEnumerable<string> _referenceImageTypes;

    static PatientImageDetails()
    {
        _referenceImageBodyParts = PatientImageDetailManager.LoadReferenceImageBodyParts();
        _referenceImageFragmentCounts = PatientImageDetailManager.LoadReferenceImageFragmentCounts();
        _referenceImageReasons = PatientImageDetailManager.LoadReferenceImageReasons();
        _referenceImageTypes = PatientImageDetailManager.LoadReferenceImageTypes();
    }

    public PatientImageDetails()
    {
        _patientImageDetailManager = new PatientImageDetailManager();
        _patientImageDetailManager.BOSaveSuccess += new BaseBO.BOSaveSuccessEvent(manager_BOSaveSuccess);
    }

    protected void Page_Load(object sender, EventArgs e)
    {
        if (ViewState[OriginallyNavigatedFrom] == null)
            ViewState[OriginallyNavigatedFrom] = Request.UrlReferrer;

        if (!IsPostBack)
        {
            int patientId;
            var patientIdPresent = int.TryParse(Request.QueryString["patientId"], out patientId);

            int patientImageId;
            var patientImageIdPresent = int.TryParse(Request.QueryString["patientImageId"], out patientImageId);

            if (patientIdPresent && patientImageIdPresent)
            {
                PopulateControls();

                var patientSummary = GetOrStorePatientSummary();
                SetBreadCrumbLabel(patientSummary);

                LoadData(patientId, patientImageId);
            }
        }
    }

    private void PopulateSpecializedRadComboBoxes()
    {
        foreach (var type in _referenceImageTypes)
            RadComboBoxImageType.Items.Add(type);

        foreach (var type in _referenceImageBodyParts)
            RadComboBoxBodyPartImaged.Items.Add(type);

        foreach (var type in _referenceImageReasons)
            RadComboBoxImagingReasons.Items.Add(type);

        foreach (var type in _referenceImageFragmentCounts)
            RadComboBoxFragmentCount.Items.Add(type);
    }

    private void PopulateBooleanRadComboBoxes()
    {
        foreach (var combo in GetBooleanRadComboBoxes())
        {
            combo.Items.Add(new RadComboBoxItem("(Select a Value)"));
            combo.Items.Add(new RadComboBoxItem("Yes"));
            combo.Items.Add(new RadComboBoxItem("No"));
        }
    }

    private IEnumerable<RadComboBox> GetBooleanRadComboBoxes()
    {
        return new List<RadComboBox>
        {
            RadComboBoxFragmentDocumented,
            RadComboBoxIncludeInReport,
            RadComboBoxReviewed
        };
    }

    private void GetReadOnlyValues(PatientImageDetail detail)
    {
        LabelStudyDescriptionData.Text = detail.StudyDescription;

        LabelSystemIDFragmentData.Text = detail.SystemIdentified.HasValue ? detail.SystemIdentified.Value ? "Yes" : "No" : "TBD";
    }

    private void SetBreadCrumbLabel(PatientSummary patientSummary)
    {
        LabelBreadCrumb.Text = $"{LabelBreadCrumb.Text}{patientSummary.FullName}";
        LabelPatientNameData.Text = patientSummary.FullName;

        LabelPatientSsnData.Text = patientSummary.Snum;
    }

    private void GetCheckedImageBodyParts(int patientImageId)
    {
        RadComboBoxBodyPartImaged.ClearCheckedItems();

        var bodyParts = _patientImageDetailManager.GetImageBodyParts(patientImageId);
        if (bodyParts != null && bodyParts.Any())
        {
            foreach (var bp in bodyParts)
                RadComboBoxBodyPartImaged.FindItemByText(bp.Description).Checked = true;
        }
    }

    private void GetCheckedImageReasons(int patientImageId)
    {
        RadComboBoxImagingReasons.ClearCheckedItems();

        var imageReasons = _patientImageDetailManager.GetImageReasons(patientImageId);
        if (imageReasons != null && imageReasons.Any())
        {
            foreach (var reason in imageReasons)
                RadComboBoxImagingReasons.FindItemByText(reason.Description).Checked = true;
        }
    }

    private void GetSelectedImageType(PatientImageDetail detail)
    {
        RadComboBoxImageType.ClearSelection();

        if (detail.ImageTypeId.HasValue)
            RadComboBoxImageType.FindItem(i => i.Index == detail.ImageTypeId.Value).Selected = true;
    }

    private void GetSelectedFragmentCount(PatientImageDetail detail)
    {
        RadComboBoxFragmentCount.ClearSelection();

        if (detail.ImageFragmentCountId.HasValue)
            RadComboBoxFragmentCount.FindItem(i => i.Index == detail.ImageFragmentCountId.Value).Selected = true;
    }

    private void LoadOtherControls(PatientImageDetail detail)
    {
        // Fragment Verified
        if (detail.FragmentVerified.HasValue)
            RadComboBoxFragmentDocumented.SelectedIndex = detail.FragmentVerified.Value ? 1 : 2;
        else
            RadComboBoxFragmentDocumented.SelectedIndex = 0;

        // Include in Report
        if (detail.IncludeInReport.HasValue)
            RadComboBoxIncludeInReport.SelectedIndex = detail.IncludeInReport.Value ? 1 : 2;
        else
            RadComboBoxIncludeInReport.SelectedIndex = 0;

        // Largest Fragment Length
        if (detail.LargestFragmentLength != null)
            RadTextBoxLargestFragmentLength.Text = detail.LargestFragmentLength.ToString();
        else
            RadTextBoxLargestFragmentLength.Text = string.Empty;

        // Largest Fragment Width
        if (detail.LargestFragmentWidth != null)
            RadTextBoxLargestFragmentWidth.Text = detail.LargestFragmentWidth.ToString();
        else
            RadTextBoxLargestFragmentLength.Text = string.Empty;

        // Smallest Fragment Length
        if (detail.SmallestFragmentLength != null)
            RadTextBoxSmallestFragmentLength.Text = detail.SmallestFragmentLength.ToString();
        else
            RadTextBoxSmallestFragmentWidth.Text = string.Empty;

        // Smallest Fragment Width
        if (detail.SmallestFragmentWidth != null)
            RadTextBoxSmallestFragmentWidth.Text = detail.SmallestFragmentWidth.ToString();
        else
            RadTextBoxSmallestFragmentWidth.Text = string.Empty;

        // Reviewed
        if (detail.Reviewed.HasValue)
            RadComboBoxReviewed.SelectedIndex = detail.Reviewed.Value ? 1 : 2;
        else
            RadComboBoxReviewed.SelectedIndex = 0;

        // Comments
        if (detail.Comments != null)
            RadTextBoxComments.Text = detail.Comments;
        else
            RadTextBoxComments.Text = string.Empty;

        // Image Reason Other Text
        if (detail.ImageReasonOtherText != null)
            RadTextBoxOtherImageReason.Text = detail.ImageReasonOtherText;
        else
            RadTextBoxOtherImageReason.Text = string.Empty;

        // Image Type  Other Text
        if (detail.ImageTypeOtherText != null)
            RadTextBoxOtherImageType.Text = detail.ImageTypeOtherText;
        else
            RadTextBoxOtherImageType.Text = string.Empty;
    }

    private void PopulateControls()
    {
        PopulateBooleanRadComboBoxes();
        PopulateSpecializedRadComboBoxes();
    }

    private bool? GetRadComboBoxBooleanValue(RadComboBox combo)
    {
        return combo.SelectedIndex > 0 ? combo.SelectedIndex == 1 : (bool?)null;
    }

    private int? GetRadComboBoxSelectedIndex(RadComboBox combo)
    {
        int? result = null;

        if (combo.SelectedIndex > 0)
            result = combo.SelectedIndex;

        return result;
    }

    private string GetValueFromRadTextBox(RadTextBox textBox)
    {
        return string.IsNullOrEmpty(textBox.Text) ? null : textBox.Text;
    }

    private decimal? GetDecimalValueFromRadTextBox(RadTextBox textBox)
    {
        return string.IsNullOrEmpty(textBox.Text) ? (decimal?)null : decimal.Parse(textBox.Text);
    }

    protected void SaveButton_Click(object sender, EventArgs e)
    {
        if (Page.IsValid)
        {
            int patientId;
            var patientIdPresent = int.TryParse(Request.QueryString["patientId"], out patientId);

            int patientImageId;
            var patientImageIdPresent = int.TryParse(Request.QueryString["patientImageId"], out patientImageId);

            if (patientIdPresent && patientImageIdPresent)
            {
                var detail = _patientImageDetailManager.GetPatientImageDetails(patientImageId);
                if (detail != null)
                {
                    _patientImageDetailManager.UpdatePatientImage(
                        GetUpdateParameters(detail.StudyId),
                        GetCheckedItemsIndexList(RadComboBoxBodyPartImaged),
                        GetCheckedItemsIndexList(RadComboBoxImagingReasons));

                    LoadData(patientId, patientImageId);
                }
            }
        }
    }

    private object[] GetUpdateParameters(string study)
    {
        return new object[]
        {
            study,
            GetRadComboBoxSelectedIndex(RadComboBoxFragmentCount),
            GetRadComboBoxSelectedIndex(RadComboBoxImageType),
            GetRadComboBoxBooleanValue(RadComboBoxFragmentDocumented),
            GetRadComboBoxBooleanValue(RadComboBoxReviewed),
            GetRadComboBoxBooleanValue(RadComboBoxIncludeInReport),
            GetDecimalValueFromRadTextBox(RadTextBoxLargestFragmentLength),
            GetDecimalValueFromRadTextBox(RadTextBoxLargestFragmentWidth),
            GetDecimalValueFromRadTextBox(RadTextBoxSmallestFragmentLength),
            GetDecimalValueFromRadTextBox(RadTextBoxSmallestFragmentWidth),
            GetValueFromRadTextBox(RadTextBoxComments),
            GetValueFromRadTextBox(RadTextBoxOtherImageReason),
            GetValueFromRadTextBox(RadTextBoxOtherImageType),
            CurrentUser.UserName
        };
    }

    private IReadOnlyList<int> GetCheckedItemsIndexList(RadComboBox combo)
    {
        List<int> items = null;
        var checkedItems = combo.CheckedItems;
        if (checkedItems != null)
            items = checkedItems.Select(i => i.Index).ToList();

        return items;
    }

    private void LoadData(int patientId, int patientImageId)
    {
        var detail = _patientImageDetailManager.GetPatientImageDetails(patientImageId);
        if (detail != null)
        {
            GetReadOnlyValues(detail);

            GetSelectedImageType(detail);
            GetSelectedFragmentCount(detail);

            GetCheckedImageBodyParts(patientImageId);
            GetCheckedImageReasons(patientImageId);

            LoadOtherControls(detail);
        }

        // We do this so that saved changes are visible when we go "back"
        BackButton.PostBackUrl = $"PatientImaging.aspx?patientId={patientId}";
    }
}